/*
    UNICODE2ANSI: questa utility converte in formato ANSI standard un file
    di testo con codifica UNICODE (UTF-16 little endian oppure big endian
    con intestazione). Nel caso in cui il file sorgente non abbia intestazione
    viene usata la codifica UNICODE UTF-16 little endian.
    Il file sorgente viene letto e riscritto prendendo soltanto gli 8 bit
    meno significativi della codifica UNICODE a 16 bit. Gli argomenti del
    programma sono i nomi dei file di origine e di destinazione.

    Copyright (C) 2010 Fabio Baccolini

    Questo programma  software libero;  lecito redistribuirlo o
    modificarlo secondo i termini della Licenza Pubblica Generica GNU
    come  pubblicata dalla Free Software Foundation; o la versione 2
    della licenza o (a propria scelta) una versione successiva.

    Questo programma  distribuito nella speranza che sia utile, ma
    SENZA ALCUNA GARANZIA; senza neppure la garanzia implicita di
    NEGOZIABILIT o di APPLICABILIT PER UN PARTICOLARE SCOPO. Si
    veda la Licenza Pubblica Generica GNU per avere maggiori dettagli.

    Questo programma deve essere distribuito assieme ad una copia
    della Licenza Pubblica Generica GNU; in caso contrario, se ne pu
    ottenere una scrivendo alla Free Software Foundation, Inc., 59
    Temple Place, Suite 330, Boston, MA 02111-1307 USA

    L'autore del software:
    Fabio Baccolini
    E-mail: fabio.baccolini@alice.it
*/

// Direttive specifiche del linguaggio di programmazione usato
#ifdef __BORLANDC__
  #pragma argsused
#endif

// Librerie da caricare
#include <string.h>
#include <stdio.h>
#include <conio.h>

// Definizione dei simboli
#define SCONOSCIUTO -1
#define BIG_ENDIAN 0
#define LITTLE_ENDIAN 1

// Prototipi di funzioni
void scambio(char &ch1, char &ch2);

// Funzione main
int main(int argc, char **argv){

// Variabili della funzione main
int formato = SCONOSCIUTO; // Formato della codifica del file sorgente
char sorgente[256]; // Nome del file sorgente
char destinazione[256]; // Nome del file destinazione
FILE *file_sorgente; // Puntatore al file sorgente
FILE *file_destinazione; // Puntatore al file di destinazione
char ch1, ch2; // ch1 byte meno significativo, ch2 byte pi significativo

if(argc!=3){
  printf("\n");
  printf("Copyright (C) 2010, Fabio Baccolini\n\n");
  printf("Questa utility converte in formato ANSI standard un file\n");
  printf("di testo con codifica Unicode con intestazione. Il file\n");
  printf("sorgente viene letto e riscritto considerando gli 8 bit\n");
  printf("meno significativi della codifica Unicode a 16 bit. Gli\n");
  printf("argomenti del programma sono i nomi dei file di origine\n");
  printf("e di destinazione.\n\n");
  printf("Esempio: unicode2ansi origine_unicode.txt destinazione_ansi.txt\n");
  return 1;
 }

strcpy(sorgente, argv[1]);
strcpy(destinazione, argv[2]);

if( strcmp(sorgente,destinazione)==0 ){
  printf("\n");
  printf("Specificare, per il file di destinazione, un nome diverso da quello\n");
  printf("di origine\n");
  return 1;
 }

// Apertura del file sorgente in modalit binaria
 if( (file_sorgente=fopen(sorgente,"rb+"))==NULL ){
  printf("\n");
  printf("Impossibile aprire il file %s\n", sorgente);
  return 1;
 }

/* Verifica che l'intestazione del file sorgente corrisponda al formato UNICODE
   UTF-16.
   - big endian 0xFE 0xFF
   - little endian 0xFF 0xFE
   Nella codifica little endian gli 8 bit meno significativi vengono anteposti. */

/* La funzione fgetc restituisce il carattere letto oppure il valore EOF
   (tipicamente -1) in caso di errore o fine file. Il tentativo di leggere un
   carattere quando il file  finito fa restituire a feof() un valore non
   nullo. */

ch1=fgetc(file_sorgente);
ch2=fgetc(file_sorgente);
if( !feof(file_sorgente)&&(ch1==(char) 0xFF)&&(ch2==(char) 0xFE) ){
  formato=LITTLE_ENDIAN;
 }
if( !feof(file_sorgente)&&(ch1==(char) 0xFE)&&(ch2==(char) 0xFF) ){
  formato=BIG_ENDIAN;
 }

// Creazione del file di destinazione in modalit binaria
if( (file_destinazione=fopen(destinazione,"wb+"))==NULL ){
  printf("\n");
  printf("Impossibile creare il file %s\n", destinazione);
  fclose(file_sorgente);
  return 1;
 }

if( formato==LITTLE_ENDIAN ){
  printf("\n");
  printf("Il file %s presenta un'intestazione Unicode Little-Endian\n", sorgente);
 }
if( formato==BIG_ENDIAN ){
  printf("\n");
  printf("Il file %s presenta un'intestazione Unicode Big-Endian\n", sorgente);
 }
if( formato==SCONOSCIUTO ){
  printf("\n");
  printf("Il file %s non presenta un'intestazione Unicode;\n", sorgente);
  printf("verra' usata la transcodifica Unicode Little-Endian\n");
  rewind(file_sorgente);
 }

/* Vengono letti i 16 bit della codifica UNICODE.
   Nel caso in cui il carattere UNICODE sia convertibile (gli 8 bit pi
   significativi sono 0x00), vengono scritti sul file di destinazione gli 8 bit
   meno significativi. */

while( !feof(file_sorgente) ){

  ch1=fgetc(file_sorgente);
  ch2=fgetc(file_sorgente);

  /* Trattasi di una ulteriore verifica per accertarsi che il numero di byte che
     compongono il file di origine sia pari, ossia che il file sia composto da
     multipli di 16 bit */
  if( (ch1!=EOF)&&(ch2==EOF) ){
    printf("\n");
    printf("Il file %s non risulta integralmente costituito\n", sorgente);
    printf("da caratteri codificati a 16 bit\n");
    fclose(file_sorgente);
    fclose(file_destinazione);
    return 1;
   }

  if( formato==BIG_ENDIAN ) scambio(ch1, ch2);

  // Se il carattere  convertibile, scrive gli 8 bit meno significativi
  if( ch2==(char) 0x00 ) fputc(ch1, file_destinazione);

 } // while

fclose(file_sorgente);
fclose(file_destinazione);
printf("\n");
printf("Conversione completata\n");
return 0;

} // main

// Implementazione delle funzioni dichiarate all'inizio
void scambio(char &ch1, char &ch2){
  char ch;
  ch=ch1;
  ch1=ch2;
  ch2=ch;
}

/*
TRATTO DA WIKIPEDIA.IT
http://it.wikipedia.org

Codifica UTF-16
===============

In UTF-16 i primi 65536 caratteri di Unicode vengono rappresentati con il loro
valore numerico, espresso in 16 bit. I caratteri al di fuori del Basic Multilingual
Plane, il cui codice  superiore a 65535 (0xFFFF) e non rappresentabile direttamente
con 16 bit, vengono rappresentati con una coppia surrogata, ovvero una coppia di
codici nell'intervallo da 0xD800 a 0xDFFF. Per esempio:

* il carattere "A", corrispondente al codice 65 (0x41), viene rappresentato
  come 0x0041

* il carattere 0x10000 diventa la coppia 0xD800, 0xDC00

* il carattere 0x10FFFD, corrispondente al limite superiore di Unicode, viene
  rappresentato con la sequenza 0xDBFF, 0xDFFD.

Unicode non assegna a nessun carattere un valore compreso tra 0xD800 e 0xDFFF,
evitando in questo modo che i singoli elementi di una coppia surrogata possano
essere confusi con un carattere Unicode valido.

Varianti di UTF-16
==================

I codici UTF-16 vengono memorizzati come parole di 16 bit, una parola per codice.
Dato che l'ordine dei byte in una parola varia a seconda dell'architettura del
calcolatore, UTF-16 prevede tre schemi di codifica, UTF-16, UTF-16LE (Little Endian)
e UTF-16BE (Big Endian).

La codifica UTF-16 impone che l'ordine dei byte venga dichiarato esplicitamente
aggiungendo un Byte Order Mark come prefisso di ogni testo codificato. Il BOM 
la forma codificata del carattere Zero width, non breaking space, corrispondente
all'esadecimale 0xFEFF, rappresentato come 0xFE,0xFF sui sistemi big endian e
0xFF, 0xFE sui sistemi little endian.

Le codifiche UTF-16BE e UTF-16LE sono identiche alla codifica UTF-16, con
l'eccezione che l'ordine dei byte  implicito, big endian per UTF-16BE, e little
endian per UTF-16LE. Il carattere 0xFEFF all'inizio di un testo rappresentato
con una di queste due codifiche viene considerato parte del testo anziche come
BOM.

La IANA ha approvato i nomi UTF-16, UTF-16BE e UTF-16LE, indifferentemente
maiuscoli o minuscoli, per l'uso su Internet. I nomi UTF_16 o UTF16, comunemente
usati, potrebbero essere riconosciuti da specifici linguaggi di programmazione o
applicazioni, ma non sono ufficialmente validi.

UTF-16  la rappresentazione nativa del testo per le versioni di Windows basate
su NT, per il linguaggio di programmazione Java e per gli ambienti .NET e Mac
OS X Cocoa e Core.

Esempi
======

carattere Unicode            nome                          codici UTF-16 carattere
122 (esadecimale 0x7A)       Z minuscola (alfabeto latino) 007A          z
27700 (esadecimale 0x6C34)   acqua (Cinese)                6C34
119070 (esadecimale 0x1D11E) chiave di Sol                 D834 DD1E


acqua, z, chiave di Sol, codificati in UTF-16

tipo di codifica    ordine dei byte           sequenza dei byte in memoria
UTF-16LE            little endian             34 6C, 7A 00, 34 D8 1E DD
UTF-16BE            big endian                6C 34, 00 7A, D8 34 DD 1E
UTF-16              little endian, con BOM    FF FE, 34 6C, 7A 00, 34 D8 1E DD
UTF-16              big endian, con BOM       FE FF, 6C 34, 00 7A, D8 34 DD 1E
*/
